<?php
/**
 * @link http://www.yiiframework.com/
 * @copyright Copyright (c) 2008 Yii Software LLC
 * @license http://www.yiiframework.com/license/
 */

namespace yii\grid;

use Closure;
use yii\base\Object;
use yii\helpers\Html;

/**
 * Column is the base class of all [[GridView]] column classes.
 *
 * @author Qiang Xue <qiang.xue@gmail.com>
 * @since 2.0
 */
class Column extends Object
{
	/**
	 * @var GridView the grid view object that owns this column.
	 */
	public $grid;
	/**
	 * @var string the header cell content. Note that it will not be HTML-encoded.
	 */
	public $header;
	/**
	 * @var string the footer cell content. Note that it will not be HTML-encoded.
	 */
	public $footer;
	/**
	 * @var callable
	 */
	public $content;
	/**
	 * @var boolean whether this column is visible. Defaults to true.
	 */
	public $visible = true;
	public $options = [];
	public $headerOptions = [];
	/**
	 * @var array|\Closure
	 */
	public $contentOptions = [];
	public $footerOptions = [];
	/**
	 * @var array the HTML attributes for the filter cell tag.
	 */
	public $filterOptions=[];


	/**
	 * Renders the header cell.
	 */
	public function renderHeaderCell()
	{
		return Html::tag('th', $this->renderHeaderCellContent(), $this->headerOptions);
	}

	/**
	 * Renders the footer cell.
	 */
	public function renderFooterCell()
	{
		return Html::tag('td', $this->renderFooterCellContent(), $this->footerOptions);
	}

	/**
	 * Renders a data cell.
	 * @param mixed $model the data model being rendered
	 * @param mixed $key the key associated with the data model
	 * @param integer $index the zero-based index of the data item among the item array returned by [[dataProvider]].
	 * @return string the rendering result
	 */
	public function renderDataCell($model, $key, $index)
	{
		if ($this->contentOptions instanceof Closure) {
			$options = call_user_func($this->contentOptions, $model, $key, $index, $this);
		} else {
			$options = $this->contentOptions;
		}
		return Html::tag('td', $this->renderDataCellContent($model, $key, $index), $options);
	}

	/**
	 * Renders the filter cell.
	 */
	public function renderFilterCell()
	{
		return Html::tag('td', $this->renderFilterCellContent(), $this->filterOptions);
	}

	/**
	 * Renders the header cell content.
	 * The default implementation simply renders [[header]].
	 * This method may be overridden to customize the rendering of the header cell.
	 * @return string the rendering result
	 */
	protected function renderHeaderCellContent()
	{
		return trim($this->header) !== '' ? $this->header : $this->grid->emptyCell;
	}

	/**
	 * Renders the footer cell content.
	 * The default implementation simply renders [[footer]].
	 * This method may be overridden to customize the rendering of the footer cell.
	 * @return string the rendering result
	 */
	protected function renderFooterCellContent()
	{
		return trim($this->footer) !== '' ? $this->footer : $this->grid->emptyCell;
	}

	/**
	 * Renders the data cell content.
	 * @param mixed $model the data model
	 * @param mixed $key the key associated with the data model
	 * @param integer $index the zero-based index of the data model among the models array returned by [[dataProvider]].
	 * @return string the rendering result
	 */
	protected function renderDataCellContent($model, $key, $index)
	{
		if ($this->content !== null) {
			return call_user_func($this->content, $model, $key, $index, $this);
		} else {
			return $this->grid->emptyCell;
		}
	}

	/**
	 * Renders the filter cell content.
	 * The default implementation simply renders a space.
	 * This method may be overridden to customize the rendering of the filter cell (if any).
	 * @return string the rendering result
	 */
	protected function renderFilterCellContent()
	{
		return $this->grid->emptyCell;
	}
}
